# FarmTrack BioSciences - Deployment Guide

This guide explains how to deploy your FarmTrack BioSciences website to production.

## Prerequisites

- Your code pushed to a Git repository (GitHub, GitLab, etc.)
- Supabase project created at https://supabase.com
- Environment variables from Supabase dashboard

## Supabase Database Setup

Your database is already configured with the required tables:

### Tables Created
1. **orders** - Stores customer product orders
   - id (uuid)
   - name (text)
   - phone (text)
   - location (text)
   - product_name (text)
   - additional_info (text, optional)
   - created_at (timestamp)
   - status (text, defaults to 'pending')

2. **email_subscriptions** - Stores newsletter subscriptions
   - id (uuid)
   - email (text, unique)
   - subscribed_at (timestamp)
   - is_active (boolean)
   - source (text, optional)

### Security Configuration
- Row Level Security (RLS) is enabled on both tables
- Public users can INSERT orders and subscriptions
- Only authenticated admins can view and manage data
- Email addresses are unique to prevent duplicates

## Environment Variables

You need these two environment variables from your Supabase project:

1. Go to https://app.supabase.com/project/YOUR_PROJECT/settings/api
2. Copy the values:
   - **Project URL** → Use as `VITE_SUPABASE_URL`
   - **anon public key** → Use as `VITE_SUPABASE_ANON_KEY`

## Deployment Options

### Option 1: cPanel Hosting

**For traditional web hosting with cPanel:**

#### Step 1: Build Your Project Locally

On your computer, run:
```bash
npm install
npm run build
```

This creates a `dist/` folder with all your website files.

#### Step 2: Upload Files to cPanel

1. **Login to cPanel**
   - Access your hosting control panel

2. **Open File Manager**
   - Navigate to `public_html/` (or your domain's root folder)

3. **Upload Files**
   - Delete any existing files in `public_html/`
   - Upload ALL contents from your local `dist/` folder to `public_html/`
   - Make sure to upload the `.htaccess` file (enable "Show Hidden Files" if needed)

#### Step 3: Configure Environment Variables

Since cPanel doesn't support build-time environment variables, you have two options:

**Option A: Create a config.js file (Recommended)**

1. In `public_html/`, create a file named `env-config.js`:
```javascript
window.ENV = {
  VITE_SUPABASE_URL: 'your_supabase_project_url',
  VITE_SUPABASE_ANON_KEY: 'your_supabase_anon_key'
};
```

2. Add this script to your `index.html` BEFORE other scripts:
```html
<script src="/env-config.js"></script>
```

3. Update your code to read from `window.ENV` instead of `import.meta.env`

**Option B: Rebuild with hardcoded values**

1. Create a `.env.production` file locally with your Supabase credentials
2. Run `npm run build` again
3. Re-upload the new `dist/` contents

#### Step 4: Verify .htaccess Configuration

The `.htaccess` file (included in the `dist/` folder) should contain:

```apache
<IfModule mod_rewrite.c>
  RewriteEngine On
  RewriteBase /

  RewriteCond %{REQUEST_FILENAME} !-f
  RewriteCond %{REQUEST_FILENAME} !-d
  RewriteRule ^ index.html [L]
</IfModule>
```

This ensures React Router works correctly.

#### Step 5: Test Your Site

1. Visit your domain
2. Test all pages and navigation
3. Submit a test order to verify Supabase connection
4. Check that forms are working

#### Common cPanel Issues

**Problem: Pages show 404 error**
- Solution: Verify `.htaccess` file is uploaded and mod_rewrite is enabled

**Problem: Assets not loading**
- Solution: Check file permissions (644 for files, 755 for folders)

**Problem: Forms not working**
- Solution: Verify environment variables are configured correctly

### Option 2: Netlify (Recommended for Easy Deployment)

1. **Connect Repository**
   - Go to https://app.netlify.com
   - Click "Add new site" → "Import an existing project"
   - Connect your Git repository

2. **Configure Build Settings**
   - Build command: `npm run build`
   - Publish directory: `dist`
   - These are already configured in `netlify.toml`

3. **Set Environment Variables**
   - Go to Site settings → Environment variables
   - Add:
     ```
     VITE_SUPABASE_URL = your_supabase_url
     VITE_SUPABASE_ANON_KEY = your_anon_key
     ```

4. **Deploy**
   - Click "Deploy site"
   - Your site will be live in a few minutes!

### Option 2: Vercel

1. **Import Project**
   - Go to https://vercel.com
   - Click "Add New" → "Project"
   - Import your Git repository

2. **Configure Settings**
   - Framework Preset: Vite
   - Build command: `npm run build`
   - Output directory: `dist`
   - These are already configured in `vercel.json`

3. **Set Environment Variables**
   - During import or in Project Settings → Environment Variables
   - Add:
     ```
     VITE_SUPABASE_URL = your_supabase_url
     VITE_SUPABASE_ANON_KEY = your_anon_key
     ```

4. **Deploy**
   - Click "Deploy"
   - Your site will be live!

### Option 3: Cloudflare Pages

1. **Create a Pages Project**
   - Go to https://dash.cloudflare.com
   - Select "Workers & Pages" → "Create application" → "Pages"
   - Connect your Git repository

2. **Configure Build**
   - Framework preset: Vite
   - Build command: `npm run build`
   - Build output directory: `dist`

3. **Set Environment Variables**
   - Add variables in the Settings section:
     ```
     VITE_SUPABASE_URL = your_supabase_url
     VITE_SUPABASE_ANON_KEY = your_anon_key
     ```

4. **Deploy**
   - Save and deploy

### Option 4: GitHub Pages

1. **Build Locally**
   ```bash
   npm install
   npm run build
   ```

2. **Deploy to GitHub Pages**
   - Install gh-pages: `npm install -D gh-pages`
   - Add to package.json scripts:
     ```json
     "deploy": "gh-pages -d dist"
     ```
   - Run: `npm run deploy`

3. **Configure Repository**
   - Go to repository Settings → Pages
   - Source: Deploy from branch `gh-pages`
   - Root directory: `/`

**Note:** GitHub Pages doesn't support environment variables natively. You'll need to hardcode them or use GitHub Actions with secrets.

## Viewing Your Orders and Subscriptions

### Using Supabase Dashboard

1. Go to https://app.supabase.com
2. Select your project
3. Click "Table Editor" in the sidebar
4. View tables:
   - **orders** - See all customer orders
   - **email_subscriptions** - See all newsletter subscriptions

### Columns in Orders Table
- Customer name
- Phone number
- Location
- Product ordered
- Additional information
- Order date
- Status (pending/contacted/completed)

### Columns in Email Subscriptions Table
- Email address
- Subscription date
- Active status
- Source (where they subscribed from)

### Exporting Data

1. In the Table Editor, select a table
2. Click the "..." menu
3. Choose "Export as CSV" or "Export as JSON"

## Custom Domain Setup

### Netlify
1. Site settings → Domain management
2. Add custom domain
3. Follow DNS configuration instructions

### Vercel
1. Project Settings → Domains
2. Add your domain
3. Configure DNS records as shown

### Cloudflare Pages
1. Project → Custom domains
2. Add domain
3. Configure DNS (automatic if using Cloudflare DNS)

## Testing Your Deployment

After deployment, test these features:

1. **Navigation** - All pages load correctly
2. **Product Catalog** - Products display properly
3. **Order Form** - Submit a test order
4. **Email Subscription** - Subscribe with a test email
5. **Mobile Menu** - Dropdown has white background (not transparent)
6. **PWA** - Can install as app on mobile

## Viewing Order and Email Data

### Method 1: Supabase Dashboard (Easiest)
- Go to your Supabase project → Table Editor
- Click on "orders" or "email_subscriptions"
- View, search, and export data

### Method 2: SQL Queries
In Supabase SQL Editor, run:

**View all orders:**
```sql
SELECT * FROM orders ORDER BY created_at DESC;
```

**View recent subscriptions:**
```sql
SELECT * FROM email_subscriptions ORDER BY subscribed_at DESC;
```

**Count orders by product:**
```sql
SELECT product_name, COUNT(*) as order_count
FROM orders
GROUP BY product_name
ORDER BY order_count DESC;
```

## Troubleshooting

### Orders not saving
- Check environment variables are set correctly
- Verify Supabase URL and key are correct
- Check browser console for errors

### Mobile dropdown still transparent
- Clear browser cache
- Hard refresh (Ctrl+Shift+R or Cmd+Shift+R)
- Check if latest code is deployed

### Build fails
- Ensure all environment variables are set
- Check build logs for specific errors
- Verify Node.js version is 16+

## Support

For issues or questions:
1. Check browser console for errors
2. Review Supabase logs in dashboard
3. Verify all environment variables are correct
4. Check that RLS policies are enabled

## Security Notes

- Never commit the `.env` file to Git
- Only share anon key (never service_role key)
- RLS policies protect your data from unauthorized access
- Public users can only insert orders/subscriptions
- Only authenticated admins can view/manage data
