/*
  # Create FarmTrack BioSciences Database Schema

  ## Overview
  This migration creates the core database tables for FarmTrack BioSciences website:
  - Customer order collection
  - Email newsletter subscriptions

  ## New Tables

  ### `orders`
  Stores customer product orders with contact information
  - `id` (uuid, primary key) - Unique order identifier
  - `name` (text, required) - Customer full name
  - `phone` (text, required) - Customer phone number
  - `location` (text, required) - Customer location (city, county)
  - `product_name` (text, required) - Name of the ordered product
  - `additional_info` (text, optional) - Additional notes or requirements
  - `created_at` (timestamptz) - Order timestamp, defaults to now()
  - `status` (text) - Order status (pending, contacted, completed), defaults to 'pending'

  ### `email_subscriptions`
  Stores newsletter email subscriptions
  - `id` (uuid, primary key) - Unique subscription identifier
  - `email` (text, unique, required) - Subscriber email address
  - `subscribed_at` (timestamptz) - Subscription timestamp, defaults to now()
  - `is_active` (boolean) - Subscription status, defaults to true
  - `source` (text, optional) - Where the subscription came from (e.g., 'footer', 'homepage')

  ## Security
  - Enable Row Level Security (RLS) on both tables
  - Allow public INSERT access for both tables (customers can submit orders and subscribe)
  - Allow authenticated admin users to SELECT, UPDATE, and DELETE records
  - Prevent data loss by using restrictive policies

  ## Important Notes
  1. Public users can only INSERT new records (create orders and subscriptions)
  2. Only authenticated users (admins) can view and manage the data
  3. Email addresses are unique to prevent duplicate subscriptions
  4. All timestamps are automatically set on creation
  5. Orders default to 'pending' status for admin review
*/

-- Create orders table
CREATE TABLE IF NOT EXISTS orders (
  id uuid PRIMARY KEY DEFAULT gen_random_uuid(),
  name text NOT NULL,
  phone text NOT NULL,
  location text NOT NULL,
  product_name text NOT NULL,
  additional_info text,
  created_at timestamptz DEFAULT now() NOT NULL,
  status text DEFAULT 'pending' NOT NULL
);

-- Create email_subscriptions table
CREATE TABLE IF NOT EXISTS email_subscriptions (
  id uuid PRIMARY KEY DEFAULT gen_random_uuid(),
  email text UNIQUE NOT NULL,
  subscribed_at timestamptz DEFAULT now() NOT NULL,
  is_active boolean DEFAULT true NOT NULL,
  source text
);

-- Enable Row Level Security
ALTER TABLE orders ENABLE ROW LEVEL SECURITY;
ALTER TABLE email_subscriptions ENABLE ROW LEVEL SECURITY;

-- Orders table policies
CREATE POLICY "Allow public to insert orders"
  ON orders FOR INSERT
  TO anon
  WITH CHECK (true);

CREATE POLICY "Allow authenticated users to view all orders"
  ON orders FOR SELECT
  TO authenticated
  USING (true);

CREATE POLICY "Allow authenticated users to update orders"
  ON orders FOR UPDATE
  TO authenticated
  USING (true)
  WITH CHECK (true);

CREATE POLICY "Allow authenticated users to delete orders"
  ON orders FOR DELETE
  TO authenticated
  USING (true);

-- Email subscriptions table policies
CREATE POLICY "Allow public to insert email subscriptions"
  ON email_subscriptions FOR INSERT
  TO anon
  WITH CHECK (true);

CREATE POLICY "Allow authenticated users to view all subscriptions"
  ON email_subscriptions FOR SELECT
  TO authenticated
  USING (true);

CREATE POLICY "Allow authenticated users to update subscriptions"
  ON email_subscriptions FOR UPDATE
  TO authenticated
  USING (true)
  WITH CHECK (true);

CREATE POLICY "Allow authenticated users to delete subscriptions"
  ON email_subscriptions FOR DELETE
  TO authenticated
  USING (true);

-- Create indexes for better query performance
CREATE INDEX IF NOT EXISTS idx_orders_created_at ON orders(created_at DESC);
CREATE INDEX IF NOT EXISTS idx_orders_status ON orders(status);
CREATE INDEX IF NOT EXISTS idx_orders_product_name ON orders(product_name);
CREATE INDEX IF NOT EXISTS idx_email_subscriptions_email ON email_subscriptions(email);
CREATE INDEX IF NOT EXISTS idx_email_subscriptions_created_at ON email_subscriptions(subscribed_at DESC);